#include "TerrainDepthShader.h"

TerrainDepthShader::TerrainDepthShader(ID3D11Device* device, HWND hwnd) : BowerShader(device, hwnd)
{
	initShader(L"terrainDepth_vs.cso", L"terrainDepth_hs.cso", L"terrainDepth_ds.cso", L"terrainDepth_ps.cso");
}

TerrainDepthShader::~TerrainDepthShader()
{
	//Release the buffers
	if (matrixBuffer)
	{
		matrixBuffer->Release();
		matrixBuffer = 0;
	}

	if (dynamicTessBuffer)
	{
		dynamicTessBuffer->Release();
		dynamicTessBuffer = 0;
	}

	//Release the sampler
	if (sampleState)
	{
		sampleState->Release();
		sampleState = 0;
	}

	if (layout)
	{
		layout->Release();
		layout = 0;
	}
	
	BaseShader::~BaseShader();
}

void TerrainDepthShader::initShader(const wchar_t* vsFilename, const wchar_t* psFilename)
{
	loadVertexShader(vsFilename);
	loadPixelShader(psFilename);

	//Setup the matrix buffer
	D3D11_BUFFER_DESC matrixBufferDesc;
	matrixBufferDesc.Usage = D3D11_USAGE_DYNAMIC;
	matrixBufferDesc.ByteWidth = sizeof(MatrixBufferType);
	matrixBufferDesc.BindFlags = D3D11_BIND_CONSTANT_BUFFER;
	matrixBufferDesc.CPUAccessFlags = D3D11_CPU_ACCESS_WRITE;
	matrixBufferDesc.MiscFlags = 0;
	matrixBufferDesc.StructureByteStride = 0;
	renderer->CreateBuffer(&matrixBufferDesc, NULL, &matrixBuffer);

	//Setup the dynamic tessellation buffer
	D3D11_BUFFER_DESC dynamicTessBufferDesc;
	dynamicTessBufferDesc.Usage = D3D11_USAGE_DYNAMIC;
	dynamicTessBufferDesc.ByteWidth = sizeof(DynamicTessType);
	dynamicTessBufferDesc.BindFlags = D3D11_BIND_CONSTANT_BUFFER;
	dynamicTessBufferDesc.CPUAccessFlags = D3D11_CPU_ACCESS_WRITE;
	dynamicTessBufferDesc.MiscFlags = 0;
	dynamicTessBufferDesc.StructureByteStride = 0;
	renderer->CreateBuffer(&dynamicTessBufferDesc, NULL, &dynamicTessBuffer);

	//Setup the sampler
	D3D11_SAMPLER_DESC samplerDesc;
	samplerDesc.Filter = D3D11_FILTER_ANISOTROPIC;
	samplerDesc.AddressU = D3D11_TEXTURE_ADDRESS_WRAP;
	samplerDesc.AddressV = D3D11_TEXTURE_ADDRESS_WRAP;
	samplerDesc.AddressW = D3D11_TEXTURE_ADDRESS_WRAP;
	samplerDesc.MipLODBias = 0.0f;
	samplerDesc.MaxAnisotropy = 1;
	samplerDesc.ComparisonFunc = D3D11_COMPARISON_ALWAYS;
	samplerDesc.MinLOD = 0;
	samplerDesc.MaxLOD = D3D11_FLOAT32_MAX;
	renderer->CreateSamplerState(&samplerDesc, &sampleState);
}

void TerrainDepthShader::initShader(const wchar_t* vsFilename, const wchar_t* hsFilename, const wchar_t* dsFilename, const wchar_t* psFilename)
{
	initShader(vsFilename, psFilename);
	loadHullShader(hsFilename);
	loadDomainShader(dsFilename);
}


void TerrainDepthShader::setShaderParameters(ID3D11DeviceContext* deviceContext, const XMMATRIX &worldMatrix, const XMMATRIX &viewMatrix, const XMMATRIX &projectionMatrix, XMFLOAT3 cameraPos, float minTessFactor, float maxTessFactor, ID3D11ShaderResourceView* displacementTexture)
{
	D3D11_MAPPED_SUBRESOURCE mappedResource;
	
	XMMATRIX tworld = XMMatrixTranspose(worldMatrix);
	XMMATRIX tview = XMMatrixTranspose(viewMatrix);
	XMMATRIX tproj = XMMatrixTranspose(projectionMatrix);

	//Map the matrix buffer
	deviceContext->Map(matrixBuffer, 0, D3D11_MAP_WRITE_DISCARD, 0, &mappedResource);
	MatrixBufferType* matrixPtr = (MatrixBufferType*)mappedResource.pData;
	matrixPtr->world = tworld;
	matrixPtr->view = tview;
	matrixPtr->projection = tproj;
	deviceContext->Unmap(matrixBuffer, 0);
	deviceContext->VSSetConstantBuffers(0, 1, &matrixBuffer);
	deviceContext->DSSetConstantBuffers(0, 1, &matrixBuffer);
	deviceContext->DSSetShaderResources(0, 1, &displacementTexture);
	deviceContext->DSSetSamplers(0, 1, &sampleState);

	//Map the dynamic tessellation buffer
	deviceContext->Map(dynamicTessBuffer, 0, D3D11_MAP_WRITE_DISCARD, 0, &mappedResource);
	DynamicTessType* dynamicPtr = (DynamicTessType*)mappedResource.pData;
	dynamicPtr->cameraPos = cameraPos;
	dynamicPtr->minTessFactor = minTessFactor;
	dynamicPtr->maxTessFactor = maxTessFactor;
	deviceContext->Unmap(dynamicTessBuffer, 0);
	deviceContext->HSSetConstantBuffers(0, 1, &dynamicTessBuffer);
}


